from django.db import models
from django.utils import timezone
from django.utils.text import slugify
from django.contrib.auth.models import User
from django_ckeditor_5.fields import CKEditor5Field 

from .calendar import Calendar

class Minutes(models.Model):
    agenda = models.TextField(help_text="Agenda of the meeting")
    minutes = CKEditor5Field('Minutes of the meeting', config_name='extends')
    meeting_date = models.DateTimeField(help_text="Date and time of the meeting")
    meeting_title = models.CharField(max_length=255, help_text="Title of the meeting")
    attendees = models.TextField(help_text="List of attendees (comma-separated or as a text block)")
    aobs = models.TextField(blank=True, help_text="Any Other Business (AOBs) discussed in the meeting")
    previous_meeting_seconded_by = models.CharField(max_length=255, blank=True, help_text="Person who seconded the previous meeting's minutes")
    members_absent_with_apologies = models.TextField(blank=True, help_text="List of members absent with apologies (comma-separated)")
    members_absent_without_apologies = models.TextField(blank=True, help_text="List of members absent without apologies (comma-separated)")
    file = models.ManyToManyField('SupportDocument', blank=True, help_text='Upload Attach any documents shares/used in this meeting')
    created_by = models.ForeignKey('UserProfile', on_delete=models.SET_NULL, null=True, help_text="User who created the minutes")
    created_at = models.DateTimeField(default=timezone.now, help_text="Date when the minutes were created")
    updated_at = models.DateTimeField(auto_now=True, help_text="Date when the minutes were last updated")
    slug = models.SlugField(unique=True, blank=True, help_text="Slug for clean URLs")

    class Meta:
        verbose_name = 'Najimudu Meeting Minute'
        verbose_name_plural = 'Najimudu Meeting Minutes'
        ordering = ['-meeting_date']

    def save(self, *args, **kwargs):
        if not self.slug:
            base_slug = slugify(f"{self.meeting_title}-{self.meeting_date.strftime('%Y-%m-%d')}")
            slug = base_slug
            counter = 1
            while Minutes.objects.filter(slug=slug).exists():
                slug = f"{base_slug}-{counter}"
                counter += 1
            self.slug = slug

        super().save(*args, **kwargs)

        description = f"Agenda: {self.agenda}\n\nMinutes: {self.minutes}" 
        
        calendar_event, created = Calendar.objects.get_or_create(
            title=self.meeting_title,
            start=self.meeting_date,
            end=self.meeting_date + timezone.timedelta(hours=1),
            color_code='Primary',  
            badge='Activity',  
            description=description,
        )

    def __str__(self):
        return f"{self.meeting_title} ({self.meeting_date.strftime('%Y-%m-%d')})"

