from django.db import models
from django.utils.text import slugify



class OrganizationType(models.Model):
    class OrganizationTypeEnum(models.TextChoices):
        NGO = 'NGO', 'Non-Governmental Organization'
        CHARITY = 'CHARITY', 'Charity'
        FOUNDATION = 'FOUNDATION', 'Foundation'
        GOVERNMENT_AGENCY ="GOVERNMENT_AGENCY", "Government Agency"
        TRUST = 'TRUST', 'Trust'
        INSTITUTE = 'INSTITUTE', 'Institute'
        ASSOCIATION = 'ASSOCIATION', 'Association'
        COOPERATIVE = 'COOPERATIVE', 'Cooperative'
        SOCIAL_ENTERPRISE = 'SOCIAL_ENTERPRISE', 'Social Enterprise'
        ADVOCACY_GROUP = 'ADVOCACY_GROUP', 'Advocacy Group'
        RESEARCH_ORG = 'RESEARCH_ORG', 'Research Organization'
        CBO = 'CBO', 'Community-Based Organization'
        SOCIETY = 'SOCIETY', 'Society'
        CLUB = 'CLUB', 'Club'
        UNION = 'UNION', 'Union'
        FOR_PROFIT = 'FOR_PROFIT', 'For Profit Organizations'
        COMPANY = 'COMPANY', 'Company'
        SOLE_PROPRIETARY = 'SOLE_PROPRIETARY', 'Sole Proprietary Businesses'
        OTHER = 'OTHER', 'Other Types of Organizations Not Listed'

    name = models.CharField(
        max_length=100,
        choices=OrganizationTypeEnum.choices,
        help_text='Select the type of organization',
        unique=True
    )
    acronym = models.CharField(max_length=30, blank=True, editable=False, help_text="THIS FIELD IS AUTO FILLED")
    description = models.TextField(blank=True, editable=False, help_text="THIS FIELD IS AUTO FILLED")
    slug = models.SlugField(unique=True, editable=False, blank=True)

    class Meta:
        verbose_name = 'Default-Organization Type'
        verbose_name_plural = 'Default-Organization Types'

    def save(self, *args, **kwargs):
        if not self.slug:
            self.slug = slugify(self.name)
        
        # Map organization type names to acronyms
        acronym_mapping = {
            'NGO': 'NGO',
            'CHARITY': 'CHARITY',
            'FOUNDATION': 'FOUND',
            'GOVERNMENT_AGENCY': 'GOV_AGENCY',
            'TRUST': 'TRUST',
            'INSTITUTE': 'INST',
            'ASSOCIATION': 'ASSOC',
            'COOPERATIVE': 'COOP',
            'SOCIAL_ENTERPRISE': 'SOC_ENT',
            'ADVOCACY_GROUP': 'ADV_GRP',
            'RESEARCH_ORG': 'RES_ORG',
            'CBO': 'CBO',
            'SOCIETY': 'SOCIETY',
            'CLUB': 'CLUB',
            'UNION': 'UNION',
            'FOR_PROFIT': 'FOR_PROFIT',
            'COMPANY': 'COMPANY',
            'SOLE_PROPRIETARY': 'SOLE_PROPRIETARY',
            'OTHER': 'OTHER'
            
        }
        
        descriptions = {
            'NGO': 'Non-Governmental Organizations (NGOs) are typically non-profit entities that operate independently of government influence, focused on humanitarian and social issues.',
            'CHARITY': 'Charities are organizations that provide aid and support to those in need, often focusing on specific causes such as poverty, illness, or education.',
            'FOUNDATION': 'Foundations are organizations that provide financial support for specific causes or projects, often through grants and endowments.',
            'GOVERNMENT_AGENCY': 'Government agencies are entities controlled by a government, responsible for implementing and enforcing laws and policies.',
            'TRUST': 'Trusts are organizations that manage assets and distribute them according to specific guidelines or purposes, often related to charitable causes.',
            'INSTITUTE': 'Institutes are organizations dedicated to research, education, or specialized training in a specific field or discipline.',
            'ASSOCIATION': 'Associations are groups of individuals or organizations with a common interest or purpose, often working together to achieve shared goals.',
            'COOPERATIVE': 'Cooperatives are organizations owned and operated by their members, who share in the decision-making and benefits of the organization.',
            'SOCIAL_ENTERPRISE': 'Social enterprises are businesses that aim to achieve social, environmental, or community goals alongside generating profit.',
            'ADVOCACY_GROUP': 'Advocacy groups work to influence public policy and opinion on specific issues or causes, often through campaigns and lobbying efforts.',
            'RESEARCH_ORG': 'Research organizations focus on conducting studies and investigations to advance knowledge and understanding in various fields.',
            'CBO': 'Community-Based Organizations (CBOs) are local entities focused on addressing community needs and promoting local development.',
            'SOCIETY': 'Societies are groups formed for specific purposes or interests, often involving members who share common goals or hobbies.',
            'CLUB': 'Clubs are organizations where members gather to pursue shared interests, recreational activities, or social engagements.',
            'UNION': 'Unions are organizations that represent workers in negotiations with employers to improve working conditions and wages.',
            'FOR_PROFIT': 'For profit organizations aim to generate profit for owners and shareholders.',
            'COMPANY': 'Companies are legal entities formed to conduct business and generate profit.',
            'SOLE_PROPRIETARY': 'Sole proprietary businesses are owned and operated by a single individual, often without formal incorporation.',
            'OTHER': 'Other types of organizations that do not fit into standard categories.'
            }

        self.description = descriptions.get(self.name, 'No description available.')
        self.acronym = acronym_mapping.get(self.name, '')
        
        super().save(*args, **kwargs)

    def __str__(self):
        return self.name
